const mysql = require('mysql2/promise');
require('dotenv').config();

// Create a connection pool with better production settings
const db = mysql.createPool({
  host: process.env.DB_HOST || "localhost",
  user: process.env.DB_USER || "root",
  password: process.env.DB_PASSWORD || "",
  database: process.env.DB_NAME || "testdb",
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0,
  // Production connection settings
  acquireTimeout: 60000, // 60 seconds
  timeout: 60000, // 60 seconds
  reconnect: true,
  // Connection pool settings
  enableKeepAlive: true,
  keepAliveInitialDelay: 0,
  // SSL settings for production (if needed)
  ssl: process.env.NODE_ENV === 'production' ? { rejectUnauthorized: false } : false
});

// Enhanced connection test with retry mechanism
async function testConnection(retries = 3) {
  for (let i = 0; i < retries; i++) {
    try {
      const connection = await db.getConnection();
      console.log("✅ Connected to MySQL database");
      connection.release();
      return true;
    } catch (err) {
      console.error(`❌ Database connection attempt ${i + 1} failed:`, err.message);
      if (i === retries - 1) {
        console.error("❌ All database connection attempts failed");
        return false;
      }
      // Wait before retrying (exponential backoff)
      await new Promise(resolve => setTimeout(resolve, Math.pow(2, i) * 1000));
    }
  }
}

// Test connection on startup
testConnection();

// Enhanced query function with connection management
async function executeQuery(query, params = []) {
  let retries = 3;
  while (retries > 0) {
    try {
      const [results] = await db.query(query, params);
      return results;
    } catch (err) {
      retries--;
      console.error(`Database query error (${3 - retries}/3):`, err.message);
      
      // If it's a connection error and we have retries left, try again
      if (retries > 0 && (err.code === 'PROTOCOL_CONNECTION_LOST' || 
                          err.code === 'ECONNRESET' || 
                          err.code === 'ETIMEDOUT' ||
                          err.message.includes('closed state'))) {
        console.log(`Retrying query... (${retries} attempts left)`);
        await new Promise(resolve => setTimeout(resolve, 1000));
        continue;
      }
      
      // If no more retries or it's not a connection error, throw the error
      throw err;
    }
  }
}

// Graceful shutdown handling
process.on('SIGINT', async () => {
  console.log('Shutting down database connections...');
  await db.end();
  process.exit(0);
});

process.on('SIGTERM', async () => {
  console.log('Shutting down database connections...');
  await db.end();
  process.exit(0);
});

module.exports = { db, executeQuery };
