const express = require('express');
const { executeQuery } = require('../config/db');
const router = express.Router();

// POST API - Create a new research work
router.post('/create', async (req, res) => {
  const { project_title, keywords, project_description, key_outcomes, collaborators } = req.body;
  
  console.log('Research work creation request received:', { project_title, collaborators });
  
  try {
    // Validate required fields
    if (!project_title || !project_description) {
      return res.status(400).json({ 
        message: 'Missing required fields: project_title and project_description are required' 
      });
    }

    // Convert keywords and key_outcomes to JSON strings if they are objects
    const keywordsJson = typeof keywords === 'object' ? JSON.stringify(keywords) : keywords;
    const keyOutcomesJson = typeof key_outcomes === 'object' ? JSON.stringify(key_outcomes) : key_outcomes;

    const results = await executeQuery(
      'INSERT INTO research_works (project_title, keywords, project_description, key_outcomes, collaborators, created_at) VALUES (?, ?, ?, ?, ?, NOW())',
      [project_title, keywordsJson, project_description, keyOutcomesJson, collaborators || null]
    );
    
    console.log('Research work inserted into database:', results);
    res.status(201).json({ 
      message: 'Research work created successfully',
      research_id: results.insertId 
    });
  } catch (err) {
    console.error('Research work creation error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// GET API - Get all research works
router.get('/all', async (req, res) => {
  console.log('Get all research works request received');
  
  try {
    const results = await executeQuery(
      'SELECT id, project_title, keywords, project_description, key_outcomes, collaborators, created_at FROM research_works ORDER BY created_at DESC'
    );
    
    // Parse JSON fields back to objects (handle both string and object cases)
    const formattedResults = results.map(item => ({
      ...item,
      keywords: item.keywords ? (typeof item.keywords === 'string' ? JSON.parse(item.keywords) : item.keywords) : null,
      key_outcomes: item.key_outcomes ? (typeof item.key_outcomes === 'string' ? JSON.parse(item.key_outcomes) : item.key_outcomes) : null
    }));
    
    console.log(`Found ${results.length} research works`);
    res.json({ 
      research_works: formattedResults,
      count: results.length 
    });
  } catch (err) {
    console.error('Get all research works error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// GET API - Get research work by ID
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  
  console.log('Get research work by ID request received:', id);
  
  try {
    const results = await executeQuery(
      'SELECT id, project_title, keywords, project_description, key_outcomes, collaborators, created_at FROM research_works WHERE id = ?',
      [id]
    );
    
    if (results.length === 0) {
      console.log('No research work found for ID:', id);
      return res.status(404).json({ message: 'Research work not found' });
    }
    
    // Parse JSON fields back to objects (handle both string and object cases)
    const formattedResult = {
      ...results[0],
      keywords: results[0].keywords ? (typeof results[0].keywords === 'string' ? JSON.parse(results[0].keywords) : results[0].keywords) : null,
      key_outcomes: results[0].key_outcomes ? (typeof results[0].key_outcomes === 'string' ? JSON.parse(results[0].key_outcomes) : results[0].key_outcomes) : null
    };
    
    console.log('Research work found for ID:', id);
    res.json(formattedResult);
  } catch (err) {
    console.error('Get research work by ID error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// DELETE API - Delete research work by ID
router.delete('/:id', async (req, res) => {
  const { id } = req.params;
  
  console.log('Delete research work request received for ID:', id);
  
  try {
    // First check if the research work exists
    const checkResults = await executeQuery(
      'SELECT id FROM research_works WHERE id = ?',
      [id]
    );
    
    if (checkResults.length === 0) {
      console.log('No research work found for deletion with ID:', id);
      return res.status(404).json({ message: 'Research work not found' });
    }
    
    // Delete the research work
    const results = await executeQuery(
      'DELETE FROM research_works WHERE id = ?',
      [id]
    );
    
    console.log('Research work deleted successfully with ID:', id);
    res.json({ 
      message: 'Research work deleted successfully',
      deleted_id: id 
    });
  } catch (err) {
    console.error('Delete research work error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = router;
