const express = require('express');
const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const { executeQuery } = require('../config/db');
const router = express.Router(); // Initialize router

// Signup
router.post('/signup', async (req, res) => {
  const { username, email, password, role } = req.body;
  console.log('Signup request received:', { username, email, password, role });
  try {
    const hashedPassword = await bcrypt.hash(password, 10);
    console.log('Password hashed successfully');
    const userRole = role === 'admin' ? 'admin' : 'user';
    
    const results = await executeQuery(
      'INSERT INTO users (username, email, password, role) VALUES (?, ?, ?, ?)',
      [username, email, hashedPassword, userRole]
    );
    
    console.log('User inserted into database:', results);
    res.status(201).json({ message: 'User created successfully' });
  } catch (err) {
    console.error('Signup error:', err);
    if (err.code === 'ER_DUP_ENTRY') {
      return res.status(400).json({ message: 'Username or email already exists' });
    }
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Login
router.post('/login', async (req, res) => {
  const { email, password } = req.body;
  console.log('Login request received:', { email });
  
  try {
    const results = await executeQuery('SELECT * FROM users WHERE email = ?', [email]);
    
    if (results.length === 0) {
      console.log('No user found for email:', email);
      return res.status(401).json({ message: 'Invalid credentials' });
    }

    const user = results[0];
    const isMatch = await bcrypt.compare(password, user.password);
    if (!isMatch) {
      console.log('Password mismatch for email:', email);
      return res.status(401).json({ message: 'Invalid credentials' });
    }

    const token = jwt.sign({ id: user.id, role: user.role }, process.env.JWT_SECRET, {
      expiresIn: '1h',
    });
    console.log('Login successful, token generated for user:', user.id);
    res.json({ token, role: user.role });
  } catch (err) {
    console.error('Login error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get user info (protected route)
router.get('/me', async (req, res) => {
  const token = req.headers.authorization?.split(' ')[1];
  if (!token) {
    console.log('No token provided in /me request');
    return res.status(401).json({ message: 'No token provided' });
  }

  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    console.log('Token verified, user ID:', decoded.id);
    
    const results = await executeQuery('SELECT id, username, email, role FROM users WHERE id = ?', [decoded.id]);
    
    if (results.length === 0) {
      console.error('No user found for token');
      return res.status(401).json({ message: 'Invalid token' });
    }
    
    res.json(results[0]);
  } catch (err) {
    console.error('Token verification error:', err);
    res.status(401).json({ message: 'Invalid token' });
  }
});

module.exports = router;