const express = require('express');
const jwt = require('jsonwebtoken');
const { executeQuery } = require('../config/db');
const router = express.Router();

// Middleware to verify JWT token
const authenticate = (req, res, next) => {
  const token = req.headers.authorization?.split(' ')[1];
  if (!token) return res.status(401).json({ message: 'No token provided' });

  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    req.user = decoded;
    next();
  } catch (err) {
    res.status(401).json({ message: 'Invalid token' });
  }
};

// Get all orders for user with search and date filters
router.get('/', authenticate, async (req, res) => {
  try {
    const { search, dateFrom, dateTo } = req.query;
    
    let query = 'SELECT * FROM orders WHERE user_id = ?';
    let params = [req.user.id];
    
    // Add search filter
    if (search) {
      query += ' AND (order_no LIKE ? OR perfume LIKE ? OR city LIKE ?)';
      const searchTerm = `%${search}%`;
      params.push(searchTerm, searchTerm, searchTerm);
    }
    
    // Add date range filter
    if (dateFrom || dateTo) {
      if (dateFrom && dateTo) {
        query += ' AND DATE(created_at) BETWEEN ? AND ?';
        params.push(dateFrom, dateTo);
      } else if (dateFrom) {
        query += ' AND DATE(created_at) >= ?';
        params.push(dateFrom);
      } else if (dateTo) {
        query += ' AND DATE(created_at) <= ?';
        params.push(dateTo);
      }
    }
    
    query += ' ORDER BY created_at DESC';
    
    const results = await executeQuery(query, params);
    res.json(results);
  } catch (err) {
    console.error('Error fetching orders:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Create order
router.post('/', authenticate, async (req, res) => {
  try {
    const { order_no, perfume, price, city } = req.body;
    const result = await executeQuery(
      'INSERT INTO orders (user_id, order_no, perfume, price, city) VALUES (?, ?, ?, ?, ?)',
      [req.user.id, order_no, perfume, price, city]
    );
    res.json({ id: result.insertId, message: 'Order created' });
  } catch (err) {
    console.error('Error creating order:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update order
router.put('/:id', authenticate, async (req, res) => {
  try {
    const { order_no, perfume, price, city } = req.body;
    const result = await executeQuery(
      'UPDATE orders SET order_no = ?, perfume = ?, price = ?, city = ? WHERE id = ? AND user_id = ?',
      [order_no, perfume, price, city, req.params.id, req.user.id]
    );
    if (result.affectedRows === 0) return res.status(404).json({ message: 'Order not found' });
    res.json({ message: 'Order updated' });
  } catch (err) {
    console.error('Error updating order:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete order
router.delete('/:id', authenticate, async (req, res) => {
  try {
    const result = await executeQuery(
      'DELETE FROM orders WHERE id = ? AND user_id = ?',
      [req.params.id, req.user.id]
    );
    if (result.affectedRows === 0) return res.status(404).json({ message: 'Order not found' });
    res.json({ message: 'Order deleted' });
  } catch (err) {
    console.error('Error deleting order:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Bulk import orders
router.post('/bulk', authenticate, async (req, res) => {
  try {
    const { orders } = req.body;
    
    if (!Array.isArray(orders) || orders.length === 0) {
      return res.status(400).json({ message: 'No orders provided' });
    }

    // Validate orders
    const validOrders = orders.filter(order => 
      order.order_no && 
      order.perfume && 
      order.price && 
      order.city
    );

    if (validOrders.length === 0) {
      return res.status(400).json({ message: 'No valid orders found' });
    }

    // Insert orders in batch
    const values = validOrders.map(order => [
      req.user.id,
      order.order_no,
      order.perfume,
      order.price,
      order.city
    ]);

    const result = await executeQuery(
      'INSERT INTO orders (user_id, order_no, perfume, price, city) VALUES ?',
      [values]
    );

    res.json({ 
      message: 'Orders imported successfully',
      insertedCount: result.affectedRows,
      totalOrders: validOrders.length
    });
  } catch (err) {
    console.error('Bulk import error:', err);
    res.status(500).json({ message: 'Server error during bulk import' });
  }
});

module.exports = router;
