const express = require('express');
const router = express.Router();
const { executeQuery } = require('../config/db');

// ===== EXPENSES =====

// Get all expenses
router.get('/expenses', async (req, res) => {
  try {
    const results = await executeQuery('SELECT * FROM expenses');
    res.json(results);
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Add new expense
router.post('/expenses', async (req, res) => {
  try {
    const { amount, addedBy, date, dueTo } = req.body;
    const results = await executeQuery(
      'INSERT INTO expenses (amount, addedBy, date, dueTo) VALUES (?, ?, ?, ?)',
      [amount, addedBy, date, dueTo]
    );
    res.json({ message: 'Expense added', id: results.insertId });
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update expense
router.put('/expenses/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const { amount, addedBy, date, dueTo } = req.body;
    await executeQuery(
      'UPDATE expenses SET amount=?, addedBy=?, date=?, dueTo=? WHERE id=?',
      [amount, addedBy, date, dueTo, id]
    );
    res.json({ message: 'Expense updated' });
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete expense
router.delete('/expenses/:id', async (req, res) => {
  try {
    const { id } = req.params;
    await executeQuery('DELETE FROM expenses WHERE id=?', [id]);
    res.json({ message: 'Expense deleted' });
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// ===== CREDITS =====

router.get('/credits', async (req, res) => {
  try {
    const results = await executeQuery('SELECT * FROM credits');
    res.json(results);
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

router.post('/credits', async (req, res) => {
  try {
    const { amount, doneBy, dueTo, date } = req.body;
    const results = await executeQuery(
      'INSERT INTO credits (amount, doneBy, dueTo, date) VALUES (?, ?, ?, ?)',
      [amount, doneBy, dueTo, date]
    );
    res.json({ message: 'Credit added', id: results.insertId });
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

router.put('/credits/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const { amount, doneBy, dueTo, date } = req.body;
    await executeQuery(
      'UPDATE credits SET amount=?, doneBy=?, dueTo=?, date=? WHERE id=?',
      [amount, doneBy, dueTo, date, id]
    );
    res.json({ message: 'Credit updated' });
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

router.delete('/credits/:id', async (req, res) => {
  try {
    const { id } = req.params;
    await executeQuery('DELETE FROM credits WHERE id=?', [id]);
    res.json({ message: 'Credit deleted' });
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// ===== AMOUNTS =====

router.get('/amounts', async (req, res) => {
  try {
    const results = await executeQuery('SELECT * FROM amounts');
    res.json(results);
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

router.post('/amounts', async (req, res) => {
  try {
    const { amount, addedBy, date } = req.body;
    const results = await executeQuery(
      'INSERT INTO amounts (amount, addedBy, date) VALUES (?, ?, ?)',
      [amount, addedBy, date]
    );
    res.json({ message: 'Amount added', id: results.insertId });
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

router.put('/amounts/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const { amount, addedBy, date } = req.body;
    await executeQuery(
      'UPDATE amounts SET amount=?, addedBy=?, date=? WHERE id=?',
      [amount, addedBy, date, id]
    );
    res.json({ message: 'Amount updated' });
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

router.delete('/amounts/:id', async (req, res) => {
  try {
    const { id } = req.params;
    await executeQuery('DELETE FROM amounts WHERE id=?', [id]);
    res.json({ message: 'Amount deleted' });
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// ===== MONTHLY PROFIT =====

router.get('/monthly-profit', async (req, res) => {
  try {
    const query = `
      SELECT 
        (SELECT SUM(amount) FROM amounts) 
        - (SELECT SUM(amount) FROM expenses) 
        + (SELECT SUM(amount) FROM credits) AS monthlyProfit
    `;
    const results = await executeQuery(query);
    res.json(results[0]);
  } catch (err) {
    console.error('Database query error:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = router;
